Test_Intersections = class()

function Test_Intersections:init()
    self.title = "intersection detection"
end

function Test_Intersections:setup()   
    self.lab = PhysicsLab() 
    self.unit = WIDTH * 0.02
    -- self.setupStarted = ElapsedTime 
    self:makeSpinningRaycaster()
    self:makeOrbitingCircle()
    self:makeAABBFrame()    
    self:makeDraggableCircles()
    self:makeNothingCollideWithAnything()
    print("Drag the box and circles around to see overlap, raycasting, and AABB querying each detect bodies in their own way. PhysicsLab makes all of these easier to work with.\n\nNote: AABB rects appear when the box is pressed or intersects an AABB rect.")
end

function Test_Intersections:draw() 
    
    self:enforceMotionsAndStillnesses()
    
    background(0) 
    self:drawAllAABBsIfNeeded()
    
    --set up function to apply to any detected AABB rects 
    local numFrameContents = 0
    local frameLL, _, _, frameUR = self.lab:cornersBounding(self.aabbFrame)
    local highlightDetectedRects = function(results)
        for _, body in ipairs(results) do
            if body.name ~= "aabbFrame" then 
                self.shouldDrawAABB = true
                self.lab.drawer:drawAABBOf(body, color(154, 120, 212, 206))
            end
        end
        --so count doesn't include frame's own body:
        numFrameContents = #results - 1
    end
    --detect rects and apply function
    self.lab:applyToAABBRectsIntersecting(frameLL, frameUR, highlightDetectedRects)
    
    --set up function to grab draggable circles from results of raycast
    local raycastStart = self.spinningBase.line:getWorldPoint(self.spinningBase.line.points[2])
    local raycastEnd = self.spinningBase.line:getWorldPoint(self.spinningBase.line.points[1])     
    local draggableCircles = {} 
    local pullOutDraggables = function(resultTables) 
        for i, resultTable in ipairs(resultTables) do 
            if resultTable.body.shapeType == CIRCLE and
            resultTable.body ~= self.spinningBase and
            resultTable.body ~= self.orbitAnchor
            then
                table.insert(draggableCircles, resultTable.body)
            end
        end
    end
    --grab draggable circles from results of raycast
    self.lab:applyToResultsIntersectingLine(raycastStart, raycastEnd, pullOutDraggables)
    
    --set up function that gets any draggable circles overlapped by orbiter
    local orbiterCirclesDetected = {}
    local getDraggables =  function(detected)
        for i, body in ipairs(detected) do 
            if body.shapeType == CIRCLE and
            body ~= self.spinningBase and
            body ~= self.orbitAnchor
            then
                table.insert(orbiterCirclesDetected, body)
            end
        end
    end
    --flip 'orbiterDoesOverlap' if anything overlaps
    self.lab:applyToBodiesOverlapping(self.orbiter, getDraggables)
    
    --do all drawing (except for detected AABB rects, drawn earlier)
    self.lab:draw()
    self:drawOrangelyAndSpin(orbiterCirclesDetected)
    self:drawPinklyAndSpin(draggableCircles)
    self:drawRaycastEmphasisLine(raycastStart, raycastEnd, color(0, 165, 255))  
    self:drawFrameThickness(frameLL.x, frameLL.y, self.aabbFrame.points[3].x * 2, self.aabbFrame.points[4].y * 2, self.shouldDrawAABB)
    self:drawReports(#draggableCircles, numFrameContents, #orbiterCirclesDetected)
end

function Test_Intersections:touched(touch)
    self.touch = touch
    self.lab:touched(touch)
    if self.aabbFrame:testPoint(touch.pos) then
        self.aabbFrame.position = touch.pos
        self.shouldDrawAABB = true
    end 
    if touch.state == ENDED or touch.state == CANCELLED then
        self.shouldDrawAABB = false
    end
end

function Test_Intersections:collide(contact)
    self.lab:collide(contact)
end

function Test_Intersections:cleanup()
    self.lab:clear()
end

--functions called in setup()

function Test_Intersections:makeSpinningRaycaster()
    --make spinning circle
    self.spinningBase = self.lab:circleAt(WIDTH*0.8, HEIGHT*0.5, self.unit * 2)
    self.spinningBase.type = KINEMATIC
    self.spinningBase.name = "spinningBase"
    
    --make line indicating position and length of raycast
    self.spinningBase.lineLength = WIDTH/3
    self.spinningBase.line = self.lab:boxAt(self.spinningBase.position.x, self.spinningBase.position.y + (self.spinningBase.lineLength * 0.5), 1, self.spinningBase.lineLength)
    self.spinningBase.line.gravityScale = 0  
    self.spinningBase.line.name = "raycast line"  
    
    --weld line to spinningBase, so it rotates along with it
    local lineAnchor = self.spinningBase.position + vec2(0, self.unit * 0.6)
    self.lab:weldBodiesTogether(self.spinningBase, self.spinningBase.line, lineAnchor)
end

function Test_Intersections:makeOrbitingCircle()
    self.orbitAnchor = self.lab:circleAt(self.spinningBase.x, self.spinningBase.y, 10)
    self.orbitAnchor.gravityScale = 0 
    self.orbitAnchor.type = STATIC
    self.orbitAnchor.name = "orbitAnchor"
    local orbiterX = self.orbitAnchor.x - (self.spinningBase.lineLength * 1.2)
    self.orbiter = self.lab:circleAt(orbiterX, self.orbitAnchor.y, self.unit * 2.25)
    self.orbiter.gravityScale = 0 
    self.orbiter.name = "orbiter"
    self.lab:pointThatTwoBodiesRevolveAround(self.orbitAnchor.x, self.orbitAnchor.y, self.orbitAnchor, self.orbiter)
    self.orbiter.angularVelocity = -270
end

function Test_Intersections:makeAABBFrame()
    local sideSize = self.unit * 9
    local frameX = (WIDTH*0.8) - (WIDTH/1.8)
    local frameY = self.spinningBase.y - (sideSize * 1.2)
    self.aabbFrame = self.lab:boxAt(frameX, frameY, sideSize, sideSize)
    self.aabbFrame.type = KINEMATIC
    self.aabbFrame.name = "aabbFrame"
    self.aabbFrame.gravityScale = 0  
end

function Test_Intersections:makeDraggableCircles()
    for i = 1, 3 do
        local x = math.random(math.floor(self.aabbFrame.x * 0.85), math.floor(self.aabbFrame.x * 1.2))
        local y = self.spinningBase.y + (self.spinningBase.y - self.aabbFrame.y) * 1.1
        y = math.random(math.floor(y * 0.7), math.floor(y * 1.05))
        local draggableCircle = self.lab:circleAt(x, y, self.unit * math.random(19, 30) * 0.1)
        draggableCircle.linearDamping = 4 --has something to do with momentum
        draggableCircle.angularDamping = 5
        draggableCircle.gravityScale = 0 
        draggableCircle.name = "circle "..i
        self.lab.categoryManager:makeIgnoreAll(draggableCircle)
    end
end

function Test_Intersections:makeNothingCollideWithAnything()
    self.lab.categoryManager:makeIgnoreAll(self.aabbFrame)
    self.lab.categoryManager:makeIgnoreAll(self.spinningBase)
    self.lab.categoryManager:makeIgnoreAll(self.spinningBase.line)
end

--functions called in draw()

function Test_Intersections:enforceMotionsAndStillnesses()
    self.spinningBase.angularVelocity = 40
    self.orbiter.angularVelocity = -100
    self.aabbFrame.angularVelocity = 0
    self.aabbFrame.angle = 0
end

function Test_Intersections:drawAllAABBsIfNeeded()
    if self.shouldDrawAABB then
        self.lab:drawAllAABB(color(122, 122, 226, 116), {self.aabbFrame})
    end 
    if not self.touch or (self.touch and (self.touch.state == ENDED or self.touch.state == CANCELLED)) then
        self.shouldDrawAABB = false
    end
end

function Test_Intersections:drawRaycastEmphasisLine(start, finish, drawColor)
    pushStyle()
    strokeWidth(4.0)
    stroke(drawColor or stroke())    
    line(start.x, start.y, finish.x,  finish.y)
    popStyle()
end

function Test_Intersections:drawPinklyAndSpin(bodies)
    pushStyle()
    strokeWidth(3.0)
    for _, body in ipairs(bodies) do 
        if body.angularVelocity < 1 then 
            body.angularVelocity = 450
        else 
            body.angularVelocity = -80
        end
        noFill()
        self.lab.drawer:drawShape(body, color(255, 0, 164))
    end
    popStyle() 
end

function Test_Intersections:drawOrangelyAndSpin(bodies)
    pushStyle()
    strokeWidth(3.0)
    for _, body in ipairs(bodies) do 
        if body.angularVelocity < 1 then 
            body.angularVelocity = -450
        else 
            body.angularVelocity = 80
        end
        noFill()
        self.lab.drawer:drawShape(body, color(255, 140, 0))
    end
    popStyle() 
end

function Test_Intersections:drawFrameThickness(x, y, w, h, isDetecting)
    pushStyle()
    noFill()
    noStroke()
    if isDetecting then
        stroke(150,150,255,255)
        strokeWidth(self.unit * 0.3)
    end
    rect(x, y, w, h)
    popStyle()
end

function Test_Intersections:drawReports(numCircles, numRects, numOverlapping)
    pushStyle()
    --make the report strings
    local raycastReport = "circles detected by raycast line: "..numCircles
    local aabbReport = "aabb rectangles detected by frame: "..numRects
    local overlapReport = "circles overlapped by orbiter: "..tostring(numOverlapping)   
    --find line height
    local _, lineHeight = textSize("ABCDEFGHIJKLMNOPQRSTUVWXYZ")   
    --draw the reports 
    textMode = CENTER
    fill(124, 216, 52)
    text(overlapReport, WIDTH/2, lineHeight * 3)
    fill(150,150,255,255)
    text(aabbReport, WIDTH/2, lineHeight * 4.5)
    fill(0, 165, 255)
    text(raycastReport, WIDTH/2, lineHeight * 6)
    popStyle()
end
